def manhattan_distance(p1, p2):
    """Manhattan distanca izmedju dve tacke"""
    return abs(p1[0] - p2[0]) + abs(p1[1] - p2[1])


def find_mst_prim(points):
    """
    Prim-ov algoritam za MST.
    Vraca listu grana (i, j) gde su i,j indeksi tacaka.
    """
    n = len(points)
    if n == 0:
        return []
    
    in_mst = [False] * n
    min_dist = [float('inf')] * n
    parent = [-1] * n
    
    min_dist[0] = 0
    mst_edges = []
    
    for _ in range(n):
        # Nadjemo najblizu tacku van MST-a
        min_val = float('inf')
        u = -1
        for i in range(n):
            if not in_mst[i] and min_dist[i] < min_val:
                min_val = min_dist[i]
                u = i
        
        if u == -1:
            break
        
        in_mst[u] = True
        
        if parent[u] != -1:
            mst_edges.append((parent[u], u))
        
        # Azuriramo distance
        for v in range(n):
            if not in_mst[v]:
                dist = manhattan_distance(points[u], points[v])
                if dist < min_dist[v]:
                    min_dist[v] = dist
                    parent[v] = u
    
    return mst_edges


def grid_sample(points, epsilon):
    """
    Grid-based sampling - deli kvadrat na mrezu i uzima reprezentativne tacke.
    """
    n = len(points)
    max_points = int(18 / (epsilon * epsilon)) + 1
    
    # Ako mozemo uzeti sve tacke, uzimamo ih
    if n <= max_points:
        return list(range(n))
    
    # Grid sampling
    grid_size = epsilon
    grid = {}
    
    for i in range(n):
        x, y = points[i]
        grid_x = int(x / grid_size)
        grid_y = int(y / grid_size)
        cell = (grid_x, grid_y)
        
        if cell not in grid:
            grid[cell] = []
        grid[cell].append(i)
    
    # Iz svake celije uzimamo jednu tacku
    # Poboljsanje: uzimamo tacku najblizu centru celije
    sampled = []
    for cell, indices in grid.items():
        if len(indices) == 1:
            sampled.append(indices[0])
        else:
            # Centar celije
            cx = (cell[0] + 0.5) * grid_size
            cy = (cell[1] + 0.5) * grid_size
            
            # Nadjemo najblizu tacku centru
            best_i = indices[0]
            best_dist = manhattan_distance(points[best_i], (cx, cy))
            
            for i in indices[1:]:
                dist = manhattan_distance(points[i], (cx, cy))
                if dist < best_dist:
                    best_dist = dist
                    best_i = i
            
            sampled.append(best_i)
    
    # Ako jos uvek imamo previse, uniformno semplujemo
    if len(sampled) > max_points:
        # Sortiramo da bi uniformno uzeli
        sampled.sort()
        step = len(sampled) / max_points
        sampled = [sampled[int(i * step)] for i in range(max_points)]
    
    return sampled


def build_spanning_tree(points, indices):
    """
    Pravi spanning tree nad podskupom tacaka.
    """
    if len(indices) <= 1:
        return []
    
    subset_points = [points[i] for i in indices]
    mst_edges = find_mst_prim(subset_points)
    
    # Mapiramo nazad na originalne indekse
    result = []
    for i, j in mst_edges:
        result.append((indices[i], indices[j]))
    
    return result


def solve():
    """Glavna funkcija"""
    
    # Citanje ulaza
    n = int(input())
    epsilon = float(input())
    
    points = []
    for _ in range(n):
        line = input().split()
        x, y = float(line[0]), float(line[1])
        points.append((x, y))
    
    # Grid-based sampling
    selected_indices = grid_sample(points, epsilon)
    
    # Spanning tree nad izabranim tackama
    tree_edges = build_spanning_tree(points, selected_indices)
    
    # Stampanje rezultata
    print(len(selected_indices))
    print(' '.join(str(i + 1) for i in selected_indices))
    for u, v in tree_edges:
        print(u + 1, v + 1)


if __name__ == "__main__":
    solve()